''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'
' Simple Function generator program for the AD9833 DDS module using
' a 25MHz MCLK source
' written by Jim Rowe
' last updated 14/12/2016 at 3:50 pm
' Requires MMBasic 5.1 or later and an ILI9341 based LCD panel with touch
' together with a Micromite BackPack
  
' Note that Micromite's SPI interface uses these pins:
' SPI Function        Micromite pin
' MOSI (SPI out)      Pin 3
' MISO (SPI in)       Pin 14
' SCK (SPI clock)     Pin 25
' SS/CS-bar/LOAD      Pin 22 (used by AD9833 as FSYNC)
'
' Also note that the SPI interface of the AD9833 has the following
' requirements:
' 1. serial data on the SDATA/MOSI line is clocked into the
'    AD9833 on the FALLING edge of SCK pulses.
' 2. the FSYNC (frame sync/SS) input is active LOW, and this
'    line must be taken low at least 5ns before the first
'    falling edge of SCK. It should only be taken high again
'    at least 10ns after the last falling edge of SCK.
' 3. So for correct operation, the SPI MODE must be set for
'    2 (&B10) -- i.e., CPOL =1 and CPHA = 0.
' 
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  OPTION AUTORUN ON
  Option Explicit
  
  DIM AS FLOAT OutFreq! = 1000     ' desired output frequency in Hertz

  DIM AS INTEGER Btn = 0
  DIM AS INTEGER Ctr = 0          ' counter used for looping, etc
  DIM AS INTEGER IntFlag = 0      ' flag to show if screen has been touched
  DIM AS INTEGER FRegVal = 0      ' calc value to send to DDS freq reg
  DIM AS INTEGER WfmCode = 1      ' code used for waveform (1 = Sinewave)
  DIM AS INTEGER outData, inData  ' SPI data variables
  DIM AS INTEGER LSBFword = 0     ' 14 + 2-bit LSB freq word to send to DDS
  DIM AS INTEGER MSBFword = 0     ' 14 + 2-bit MSB freq word to send to DDS
  
Dim AS INTEGER RegData(5) ' array for AD9833 register initialise codes
RegData(0) = &H2100   ' resets all registers, sets DB13 for loading freq
'    registers via two consecutive 14-bit words (LSB first, then MSB) 
RegData(1) = &H69F1   ' the LSB for Freq0 reg and 1000Hz output
RegData(2) = &H4000   ' and the MSB for Freq0 reg and 1000Hz output
RegData(3) = &HC000   ' loads the Phase0 reg with 000h
RegData(4) = &H2000   ' exit reset mode for normal operation

DIM AS STRING CurrFreq$           ' current frequency as a string
DIM AS STRING CurrWfm$            ' current waveform as a string
DIM AS STRING NuFreq$             ' string for nominated new frequency
DIM AS STRING KPInStr$ LENGTH 9   ' input string from touch screen
DIM AS STRING KPStr$              ' input char from InCharFrmKP subroutine
  
  CONST FoutFactor! = 10.73741824 ' scaling factor for getting Fout from OutFreq!
                                  ' equals 2^28/Fmclk, where Fmclk = 25MHz
  Const TRUE = 1
  Const NULL = 0
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  SETPIN 22, DOUT         ' make pin 22 a digital output (used for AD9833 FSYNC)
  PIN(22) = 1             ' and set it high to make sure FSYNC is in idle state
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen is touched
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' Main program starts here
  InitAD9833    ' first go and initialise the AD9833
  ShowOScrn     ' then show the opening screen
  DO
    CurrFreq$ = STR$(OutFreq!)  ' get current freq as a string
    SELECT CASE WfmCode
      CASE 1
        CurrWfm$ = "Sinewave"
      CASE 2
        CurrWfm$ = "Triangular"
      CASE 3
        CurrWfm$ = "Squarewave"
    END SELECT    ' found current wfm as a string also
    
    FirstMenu   ' now display the first menu screen, get options
    SELECT CASE Btn
      CASE 1
        ChangeFreq  ' B1 touched, so call ChangeFreq sub
      CASE 2
        ChangeWfm   ' B2 touched, so call ChangeWfm sub
    END SELECT
  LOOP    ' End of main program loop  
END     ' End of program
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
SUB ShowOScrn ' sub to show the opening screen for five seconds
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 4, RGB(Cyan), DBlue
  TEXT 20, MM.VRes/4, "0", LM, 8, 1, Red, DBlue   ' this is the SC logo char/font
  TEXT MM.HRes/2, MM.VRes*3/4, "Simple AD9833", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*7/8, "Function Generator", CM, 1, 2, White, DBlue
  PAUSE 5000  ' pause for five seconds before moving on
END SUB
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
Sub FirstMenu   ' sub to display the first (top level) menu screen
  IntFlag = 0   ' clear interrupt flag to begin
  CLS Black
  ' draw the two buttons
  RBOX 0, MM.VRes/2, MM.HRes/2-4, MM.VRes/2-4, 10, Black, Bone
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-4, MM.VRes/2-4, 10, Black, Bone
  COLOUR White, DBlue
  ' show current frequency & waveform, followed by button text
  TEXT 0, MM.VRes/16, "FREQUENCY:", LM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes*5/32, CurrFreq$ + " Hz", CM, 1, 2, White, Black
  TEXT 0, MM.VRes*9/32, "WAVEFORM:", LM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes*6/16, CurrWfm$, CM, 1, 2, White, Black
  TEXT MM.HRes/4, MM.VRes*11/16, "Change", CM, 1, 2, Black, Bone
  TEXT MM.HRes/4, MM.VRes*13/16, "Frequency", CM, 1, 2, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*11/16, "Change", CM, 1, 2, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*13/16, "Waveform", CM, 1, 2, Black, Bone
FM2: IF TOUCH(X) = -1 THEN GOTO FM2   ' keep looping until user touches screen
  
  IF TOUCH(Y) < MM.VRes/2 THEN GOTO FM2 ' loop back if neither button touched
  SELECT CASE TOUCH(X)  ' then decide which button was touched
    CASE 0 TO MM.HRes/2
      Btn = 1         ' must have been first button
    CASE MM.HRes/2 TO MM.HRes
      Btn = 2         ' must have been second button
  END SELECT
END SUB   ' then return, with Btn set to touched button number (1 or 2)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' subroutine to change the DDS output frequency
 SUB ChangeFreq
  IntFlag = 0   ' clear interrupt flag to begin 
  CLS Black
  TEXT 0,MM.VRes/16, "Current Frequency:", LM,1,1, White, Black
  TEXT MM.HRes/4, MM.VRes*3/16, CurrFreq$ + " Hz", CM, 1,2, White, Black
  TEXT 0, MM.VRes*5/16, "Enter New Freq:", LM, 1,1, White, Black
  BOX 0, MM.VRes*3/8, MM.HRes/2 -4, MM.VRes/8, 0, White, White
  TEXT MM.HRes/4, MM.VRes*7/16, SPACE$(9), CM, 1,2, Red, White
  TEXT 0, MM.VRes*5/8, "TOUCH OK BUTTON",LM,1,1, Green, Black
  TEXT 0, MM.VRes*11/16, "BELOW TO ENTER:", LM,1,1,Green, Black
  KPadDraw              ' now go draw keypad
  KPInStr$ = SPACE$(9)  ' clear the input string
  DO
    InCharFrmKP   ' then go get a character/string from the keypad (> KPStr$)
    SELECT CASE KPStr$
      CASE "OK"    ' OK button was touched
        IF KPInStr$ = SPACE$(9) THEN EXIT DO  ' if KPInStr$ null string, exit now
        NuFreq$ = LEFT$(KPInStr$, 9)  ' otherwise make KPInStr$ the new Frequency
        TEXT MM.HRes/4, MM.VRes*7/16, NuFreq$, CM, 1,2, Red, White  ' show it
        CurrFreq$ = NuFreq$       ' also update CurrFreq$ 
        OutFreq! = VAL(NuFreq$)   ' then update output frequency
        UpdateDDSFreq             ' & go change AD9833 frequency to match
        EXIT DO  ' then leave
      CASE "0" TO "9", "-", "."
        KPInStr$ = RIGHT$(KPInStr$, 8) ' a numeral, so shorten KPInStr$
        KPInStr$ = KPInStr$ + KPStr$  ' then add the new digit to it
        TEXT MM.HRes/4, MM.VRes*7/16, KPInStr$, CM, 1,2, Red, White ' & show it
      CASE CHR$(08)
        KPInStr$ = Left$(KPInStr$, LEN(KPInStr$) -1) 'if it's a backspace, blank last char
        TEXT MM.HRes/4, MM.VRes*7/16, KPInStr$ + " ", CM, 1,2, Red, White '& re-show
    END SELECT
    PAUSE 250
  LOOP
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to show the 'change waveform' screen, make the change 
Sub ChangeWfm
  IntFlag = 0   ' clear interrupt flag to begin
  CLS Black
    ' draw the three horizontal buttons
  RBOX 0, MM.VRes/4, MM.HRes-4, MM.VRes/2-4, 10, Black, Bone
  RBOX 0, MM.VRes/2, MM.HRes-4, MM.VRes*3/4-4, 10, Black, Bone
  RBOX 0, MM.VRes*3/4, MM.HRes-4, MM.VRes, 10, Black, Bone
  COLOUR White, DBlue
  ' show current waveform plus select options text
  TEXT 0, MM.VRes/16, "CURRENT WAVEFORM: " + CurrWfm$, LM, 1, 1, White, Black
  TEXT 0, MM.VRes*3/16, "SELECT NEW WAVEFORM:", LM, 1, 2, White, Black
  ' followed by button text
  TEXT MM.HRes/2, MM.VRes*3/8, "SINEWAVE", CM, 1, 3, Black, Bone
  TEXT MM.HRes/2, MM.VRes*5/8, "TRIANGULAR", CM, 1, 3, Black, Bone
  TEXT MM.HRes/2, MM.VRes*7/8, "SQUAREWAVE", CM, 1, 3, Black, Bone

SM2: IF TOUCH(X) = -1 THEN GOTO SM2   ' keep looping until LCD touched
  SELECT CASE TOUCH(Y)  ' then decide which button was touched
    CASE MM.VRes/4 TO MM.VRes/2
      WfmCode = 1
      CurrWfm$ = "Sinewave"
      UpdateWfm   ' send updated waveform code to DDS
    CASE MM.VRes/2 TO MM.VRes*3/4
      WfmCode = 2
      CurrWfm$ = "Triangular"
      UpdateWfm   ' send updated waveform code to DDS
    CASE MM.VRes*3/4 TO MM.VRes
      WfmCode = 3
      CurrWfm$ = "Squarewave"
      UpdateWfm   ' send updated waveform code to DDS
    CASE ELSE
      EXIT SUB    ' to ignore touch at top of screen
  END SELECT
END SUB   ' then return
  
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to work out DDS Freq0 values from OutFreq! and then
' send to DDS as two 14-bit words (LSB first, then MSB)
SUB UpdateDDSFreq 
    FRegVal = OutFreq! * FoutFactor! ' get full Freq0 value
    LSBFword = FRegVal AND &H3FFF  ' extract the lower 14 bits
    outData = LSBFword OR &H4000  ' add 01 prefix for Freq0
    SendIt      ' and send it to the DDS chip
    MSBFword = (FRegVal >> 14) AND &H3FFF  ' now get upper 14 bits
    outData = MSBFword OR &H4000  ' again add the prefix for Freq0
    SendIt      ' and send it as well
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to send a new waveform code to DDS control register
SUB UpdateWfm 
  SELECT CASE WfmCode
      CASE 1
        outData = &H2000  ' set outData for a sinewave output
        SendIt            ' then go send it to DDS
      CASE 2
        outData = &H2002  ' set outData for a triangular wave
        SendIt            ' then go send it to DDS
      CASE 3
        outData = &H2068  ' set outData for a squarewave
        SendIt            ' then go send it to DDS
  END SELECT
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to set IntFlag when touchscreen touched
SUB TchInt
  IntFlag = 1
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to draw keypad on right, OK & backspace buttons lower left  
SUB KPadDraw  
  ' keypad boxes
  RBOX MM.HRes/2, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  ' boxes at lower left for backspace and OK buttons, plus legends
  RBOX 0, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4, 10, Black, Bone
  RBOX MM.HRes/6, MM.VRes*3/4, MM.HRes/3-4, MM.VRes/4, 10, Black, Bone
  TEXT MM.HRes/12, MM.VRes*7/8, "<-", CM, 1, 2, Black, Bone
  TEXT MM.HRes/3, MM.VRes*7/8, "OK", CM, 1, 3, Red, Bone
  ' legends for the keypad buttons
  TEXT MM.HRes*7/12, MM.VRes/8, "7", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes/8, "8", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes/8, "9", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*3/8, "4", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*3/8, "5", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*3/8, "6", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*5/8, "1", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*5/8, "2", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*5/8, "3", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*7/8, "-", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*7/8, "0", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*7/8, ".", CM, 1, 3, Black, Bone
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''  
SUB InCharFrmKP   ' sub to get a char from the touch keypad
                  ' (returns char in KPStr$)
  IntFlag = 0   ' clear interrupt flag to begin
ICFK2: IF TOUCH(X) = -1 THEN GOTO ICFK2 'wait until keypad touched
  ' but also loop back if no valid button was touched
  IF TOUCH(X) < MM.HRes/2 AND TOUCH(Y) < MM.VRes*3/4 THEN GOTO ICFK2
  SELECT CASE TOUCH(X)  ' now decide which button was touched
    CASE 0 TO MM.HRes/6-1
      KPStr$ = CHR$(08)  ' must have been backspace button
    CASE MM.HRes/6 TO MM.HRes/2-1
      KPStr$ = "OK"      ' must have been OK button
    CASE MM.HRes/2 TO MM.HRes*2/3-1  'first column of keypad itself
      SELECT CASE TOUCH(Y)  ' must be first column
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "7"   ' must be the 7 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "4"   ' must be the 4 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "1"   ' must be the 1 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "-"   ' must be the hyphen button
      END SELECT
    CASE MM.HRes*2/3 TO MM.HRes*5/6-1 'centre column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "8"   ' must be the 8 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "5"   ' must be the 5 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "2"   ' must be the 2 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "0"   ' must be the 0 button
      END SELECT
    CASE MM.HRes*5/6 TO MM.HRes   ' last column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "9"   ' must be the 9 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "6"   ' must be the 6 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "3"   ' must be the 3 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "."   ' must be the DP button
      END SELECT
  END SELECT
END SUB

'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'  subroutine to initialise the AD9833 DDS chip and then
'  start generating a 1000Hz sinewave
SUB InitAD9833
    For Ctr = 0 To 4
      outData = RegData(Ctr)
      SendIt
    Next
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to send a 16-bit word to the AD9833 via the SPI port
SUB SendIt
  SPI OPEN 5000000,2, 16  ' first open SPI (speed, mode, bits)
  PIN(22) = 0             ' then bring the FCLK line low
  inData = SPI(outData)   ' then send out the data word
  PIN(22) = 1             ' then bring FCLK high to end transaction
  SPI CLOSE               ' finally close the SPI port again
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' SCLOGOchar.bas
' Font type    : Numeric (1 characters)
' Font size    : 200x84 pixels
' Memory usage : 2100 bytes
' Font adapted from: http://www.rinkydinkelectronics.com/r_fonts.php

DefineFont #8
  013054C8 FF010000 F01F00F8 0F00E03F F8FF01F8 E0FF0300 FF00FE03 070000E0
  3F00FCFF 00C07FF0 FF07F81F FF0F00FC 00FE03F0 0000E0FF 00FEFF1F C07FF03F
  0FF81F00 1F00FEFF FE03F8FF 00E0FF00 FFFF7F00 7FF03F00 F81F00C0 00FFFF1F
  03FCFF3F E0FF01FF FFFF0100 F03F80FF 1F00C07F FFFF3FF8 FEFF7F80 FF01FF03
  FF0100E0 3FC0FFFF 00C07FE0 FF7FF01F FFFFC0FF 01FF07FF 0300E0FF E0FFC0FF
  80FFE07F 7FF03F00 FFC07FE0 FF07FF81 00C0FF01 FFC0FF03 FFE07FF0 F03F0080
  C1FFE07F 07FF83FF C0FF83FF C0FF0300 E07FE0FF 3F0080FF FFE07FF0 FF83FFC1
  FF83FF07 FF0300C0 7FE0FF80 0080FFE0 E07FF03F 83FFC1FF 83FF07FF 0300C0FF
  E0FF80FF 00FFC07F FFE03F00 FF81FFC0 FF0FFE03 0080FF83 FF81FF07 FFC1FFE0
  E07F0000 81FFC1FF 0FFE07FF 80FFC7FF 81FF0700 C1FFC0FF 7F0000FF FFC1FFE0
  FE07FF83 FFC7FF0F FF070080 FFC0FF81 0000FFC1 C1FFE07F 07FF83FF C7FF0FFE
  070080FF 000000FF 00FFC1FF FFE07F00 FF83FFC3 FF0FFE07 0080FFC7 0000FF07
  FE81FF00 C17F0000 03FF83FF 1FFC07FE 00FFEFFF 00FF0F00 83FF0100 FF0000FE
  0080FFC1 FC0FFE03 FFEFFF1F FF0F0000 FF010000 0000FE83 80FFC1FF 0FFE0700
  EFFF1FFC 0F0000FF 01E0FFFF 00FE83FF FFC1FF00 FE070080 FF1FFC0F 0000FFFF
  F0FFFF0F FE83FF01 C1FF0000 070080FF 1FFC0FFE 00FFFFFF FFFF0F00 03FF01FC
  FF0000FC 0000FF83 F80FFC07 FEFFFF3F FF070000 FF03FEFF 0100FC07 00FF83FF
  1FFC0700 FFFF3FF8 030000FE 03FFFFFF 00FC07FF FF83FF01 FC0F0000 FF3FF81F
  0000FEFF FFFFFF01 FC07FF83 83FF0100 0FFE03FF 3FF81FFC 00FEFFFF 07000000
  07FF83FF FF0100FC FE03FF83 F81FFC0F FEFFFF3F 00000000 FE83FF07 0100F807
  03FE07FF 1FF80FFE FFFF7FF0 000000FC 07FF0700 00F80FFE FE07FF03 F80FFC07
  FF7FF03F 0000FCFF FF070000 F80FFE07 07FF0300 1FFC07FE 7FF03FF8 00FCFFFB
  07FE1F00 0FFE07FF FF0300F8 FC07FE07 F03FF81F FCFFFB7F FE1F0000 FE07FF07
  0300F80F 07FE07FF 3FF81FFC FFFB7FF0 1F0000FC 07FF0FFE 00F00FFC FC0FFE03
  F01FFC07 F1FFE07F 0000F8FF FE0FFC3F F01FFC0F 0FFE0700 1FFC0FFC FFE07FF0
  00F8FFF1 0FFC3F00 1FFC0FFE FE0700F0 F80FFC0F E07FF03F F8FFF0FF FF3F0000
  FC0FFEFF 07FFFF1F 0FFC0FFE FFFF3FF8 FFF0FFE0 3F0000F8 0FFEFFFF FFFF1FFC
  FF0FFE07 FF3FF8FF E0FFE0FF 0000F8FF FCFFFF1F FF1FF80F 07FC07FF 1FF8FFFF
  FFC0FFFF 00F07FE0 FFFF0F00 3FF81FF8 FC0FFFFF F0FFFF03 81FFFF0F F07FE0FF
  FF070000 F81FE0FF 0FFFFF3F FFFF01FC FEFF07C0 7FE0FF01 030000F0 1F80FFFF
  FEFF3FF8 7F00FC0F FF0100FF C0FF01F8 0000F03F 00FFFF01 FF3FF81F 00FC0FFE
  0000FC3F FF01E07F 00E03FC0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 FEFF0100 3FF83F00 FF81FFE0 0000F8FF
  00000000 00000000 FF070000 F03F00FF 81FFE03F 00FCFFFF 00000000 00000000
  0F000000 7F80FFFF FFC17FF0 FEFFFF03 00000000 00000000 00000000 C0FFFF3F
  C17FF07F FFFF03FF 000000FF 00000000 00000000 FFFF7F00 7FF07FE0 FF03FFC1
  0080FFFF 00000000 00000000 FF7F0000 F07FF0FF 03FFC17F C0FFFFFF 00000000
  00000000 7F000000 7FE03FF0 FFC1FFE0 FF03FE03 000000C0 00000000 00000000
  E07FF0FF 83FFE0FF 03FE07FE 000080FF 00000000 00000000 7FF0FF00 FFE0FFE0
  FE07FE83 0080FF03 00000000 00000000 E0FF0000 E0FFE07F 07FE83FF 80FF03FE
  00000000 00000000 FF000000 FFE07FE0 FE83FFC1 FF03FE07 00000080 00000000
  00000000 C0FFE0FF 83FFC1FF 07FC07FE 000080FF 00000000 00000000 FFE0FF01
  FFC1FFC1 FC0FFC07 0000FF07 00000000 00000000 C0FF0100 C1FFC1FF 0FFC07FF
  00FF07FC 00000000 00000000 FF010000 FFC1FFC1 FC07FF81 FF07FC0F 00000000
  00000000 01000000 81FFC1FF 07FF83FF 07FC0FFC 000000FF 00000000 00000000
  00C0FF01 FFFFFF01 FF0FFC07 0000FFFF 00000000 00000000 80FF0300 FFFF0300
  1FF80FFE 00FEFFFF 00000000 00000000 FF030000 FF030080 F80FFEFF F8FFFF1F
  00000000 00000000 03000000 030080FF 0FFEFFFF FFFF1FF8 000000F0 00000000
  00000000 0080FF03 FEFFFF03 FF1FF80F 0000C0FF 00000000 00000000 80FF0300
  FFFF0300 1FF80FFE 0080FFFF 00000000 00000000 FF070000 FE07FF03 F01FFC0F
  0000F83F 00000000 00000000 07000000 07FF03FF 1FFC0FFE 00F83FF0 00000000
  00000000 00000000 FF03FF07 FC0FFE07 F83FF01F 00000000 00000000 00000000
  03FF0700 0FFC07FF 3FF01FFC 000000F8 00000000 00000000 FF070000 FC07FE07
  F01FFC1F 0000F83F 00000000 00000000 0F000000 0FFE07FE 3FF81FFC 00F07FE0
  00000000 00000000 00000000 FE07FE0F F81FFC0F F07FE03F 00000000 00000000
  00000000 07FE0F00 1FFC0FFE 7FE03FF8 000000F0 00000000 00000000 FE0F0000
  F80FFE07 E03FF83F 0000F07F 00000000 00000000 0F000000 0FFC0FFC 3FF83FF8
  00F07FE0 00000000 00000000 00000000 FC0FFC1F F03FF81F E0FFC07F 00000000
  00000000 00000000 0FFC1F00 3FF81FFC FFC07FF0 000000E0 00000000 00000000
  FF1F0000 F81FFCFF C07FF03F 0000E0FF 00000000 00000000 1F000000 1FF8FFFF
  7FF03FF0 00E0FFC0 00000000 00000000 00000000 F0FFFF0F F03FF01F E0FFC07F
  00000000 00000000 00000000 FFFF0700 7FF03FE0 FF81FFE0 000000C0 00000000
  00000000 FF010000 F03F80FF 81FFE07F 0000C0FF 00000000 00000000 00000000
  3F00FEFF FFE07FF0 00C0FF81 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
End DefineFont
